"""Represents the menu where the user selects a turret etc"""

import serge.blocks.actors
import serge.blocks.layout
import serge.blocks.utils

from theme import G, theme
import common 

class MainMenu(serge.blocks.actors.ScreenActor):
    """The main menu"""

    def __init__(self, layout, board):
        """Initialise the MainMenu"""
        super(MainMenu, self).__init__('menu', 'main-menu')
        self.layout = layout
        self.board = board
        
    def addedToWorld(self, world):
        """We were added to the world"""
        super(MainMenu, self).addedToWorld(world)
        self.manager = world.findActorByName('behaviours')
        self.achievements = serge.blocks.achievements.getManager()
        self.placed = set()
        self.number = 0
        #
        self.outer = serge.blocks.utils.addActorToWorld(world, 
                serge.blocks.layout.HorizontalBar('menu', 'outer-menu', G('main-menu-width'), G('main-menu-height'),
                    G('main-menu-colour'), 'ui-background'), 
                layer_name='ui', center_position=G('main-menu-position')) 
        #
        # Add the score display
        self.score_menu = serge.blocks.layout.VerticalBar('score-menu', 'score-menu',
                    G('score-menu-width'), G('score-menu-height'),
                    G('score-menu-colour'), 'ui-middleground')
        text_props = {
            'font_size' : G('score-menu-font-size'),           
            'colour' : G('score-menu-font-colour'),
            'justify' : 'center',
        }        
        self.cash = self.score_menu.addActor(
            serge.blocks.actors.NumericText('text', 'cash', 'Money\n$%04d', value=0, **text_props))
        self.base = self.score_menu.addActor(serge.actor.Actor('progress', 'base'))
        self.base.visual = serge.blocks.visualblocks.ProgressBar(
            G('base-damage-size'), G('base-damage-colours'), G('base-damage-border-width'), G('base-damage-border-colour'))
        self.base.visual.value = 100
        self.outer.addActor(self.score_menu)
        self.score_menu.setLayerName('ui-buttons')
        #
        # Add individual turrets
        for idx in range(1, 100):
            theme_name = 'turret-%d' % idx
            if not theme.hasTheme(theme_name):
                break
            #
            # Get turret properties
            name = G('name', theme_name)
            description = G('description', theme_name)
            rnge = G('range', theme_name)
            speed = G('speed', theme_name)
            damage = G('damage', theme_name)
            cost = G('cost', theme_name)
            #
            # Create turret
            item = serge.blocks.layout.VerticalBar('clickable-menu', 'turret-%d' % idx,
                    G('turret-menu-width'), G('turret-menu-height'),
                    G('turret-menu-off-colour'), 'ui-middleground')
            self.outer.addActor(item)
            #
            button = serge.actor.Actor('button', 'turret-%d-button' % idx)
            button.setSpriteName('turret-%d' % idx)
            item.addActor(button)
            #
            text_props = {
                'font_size' : G('turret-menu-font-size'),           
                'colour' : G('turret-menu-font-colour'),
            }
            #
            item.addActor(serge.blocks.actors.StringText('text', 'space', '', **text_props)) # Spacing
            item.addActor(serge.blocks.actors.StringText('text', 'name', name, **text_props))
            item.addActor(serge.blocks.actors.StringText('text', 'cost', '$%s/%d/%d/%d' % (cost, rnge, damage, speed),
                 **text_props))
            #
            item.setLayerName('ui-buttons')
            #
            item.linkEvent(serge.events.E_LEFT_CLICK, self.turretClicked, item)
            item.theme_name = theme_name
        #
        # Pop-out details
        self.pop_out = serge.blocks.layout.VerticalBar('popout-menu', 'popout-menu',
                G('popout-menu-width'), G('popout-menu-height'),
                G('turret-menu-on-colour'), 'ui')
        #
        # Add some string items to be able to show properties later
        for idx in range(6):
            self.pop_out.addActor(serge.blocks.actors.StringText('text', ('popout-item-%s' % idx), '', **text_props))
        self.pop_out.setLayerName('ui-buttons')
        self.pop_out.visible = False
        #
        world.addActor(self.pop_out)
        #
        # The selection marker
        self.selection_marker = serge.blocks.utils.addActorToWorld(world, 
            serge.actor.MountableActor('selection', 'selection'), sprite_name='turret-bad', layer_name='turrets')
        self.selection_ring = serge.blocks.utils.addVisualActorToWorld(world, 'selection', 'selection-ring',
            serge.blocks.visualblocks.Circle(100, G('selection-colour'), G('selection-width')), 'turrets')
        self.selection_marker.mountActor(self.selection_ring, (0, 0))
        self.selection_marker.active = False
        self.turret_angle = 0.0
        #
        self.outer.reflowChildren()
        self.mousedOver = serge.actor.ActorCollection()
        self.selected = serge.actor.ActorCollection()            

    def updateActor(self, interval, world):
        """Update the menu"""
        super(MainMenu, self).updateActor(interval, world)
        #
        # Unselect
        if self.mouse.isDown(serge.input.M_RIGHT):
            self.selected = None
        #
        # Reset colouring
        self.mousedOver = self.mouse.getActorsUnderMouse(world).findActorsByTag('clickable-menu')
        self.hidePopout()
        for menu in self.outer.getChildren():
            if menu == self.selected:
                colour = G('turret-menu-selected-colour')
                if menu in self.mousedOver:
                    self.showPopout(menu.theme_name, menu.x, menu.y)
            elif menu in self.mousedOver:
                colour = G('turret-menu-on-colour')
                self.showPopout(menu.theme_name, menu.x, menu.y)
            else:
                colour = G('turret-menu-off-colour')
            menu.setBackgroundColour(colour)
        #
        # Show selection
        if self.selected and self.cash.value >= G('cost', self.selected.name) and not self.mousedOver:
            #
            # Snap location to the grid
            x, y = self.mouse.getScreenPos()
            cx, cy = self.layout.getLocation((x, y))
            can_place = self.board.canPlaceAt(self.selected.name, (cx, cy))
            #
            # Find the turret rotation to a target
            targets = self.board.getTargetsFor((x, y), G('range', self.selected.name))
            if targets:
                offset = common.Vec2d(targets[0][1].x, targets[0][1].y) - common.Vec2d(x, y)
                self.turret_angle = -offset.angle_degrees
            #
            # Are we dropping - look to see if we clicked and are on a valid location
            # and we are not hovering over the menu 
            if can_place and self.mouse.isClicked(serge.input.M_LEFT) and len(self.mousedOver) == 0:
                turret = self.board.placeItem(self.selected.name, (x, y))
                if turret and G('rotates', self.selected.name):
                    turret.setAngle(self.turret_angle)
                self.cash.value -= G('cost', self.selected.name)
                self.number += 1
                self.placed.add(self.selected.name)
                self.achievements.makeReport('placed', placed_types=self.placed, number=self.number)
                serge.sound.Sounds.play('placed')
            else:
                self.selection_marker.active = True
                self.selection_marker.moveTo(x, y)
                self.selection_marker.setSpriteName(
                    self.selected.name if can_place else G('off-sprite', self.selected.name))
                self.selection_ring.active = can_place
                #
                # Rotate the turret to a target
                if G('rotates', self.selected.name):
                    self.selection_marker.setAngle(self.turret_angle)
        else:
            self.selection_marker.active = False
        
    def turretClicked(self, obj, turret):
        """A turret was clicked"""
        if self.active:
            self.log.info('You clicked on %s' % turret.name)
            self.selected = turret
            self.selection_ring.visual.radius = G('range', turret.name)
            self.selection_ring.visual.colour = G('colour', turret.name)

    def takeDamage(self, damage):
        """Take some damage"""
        self.base.visual.value -= min(self.base.visual.value, damage)
        
    def isDestroyed(self):
        """Return True if the base is destroyed"""
        return self.base.visual.value == 0

    def getHealth(self):
        """Return the current health"""
        return self.base.visual.value
                
    def addScore(self, score, cash):
        """Add to our score"""
        self.cash.value += cash
        
    def setCash(self, cash):
        """Set the amount of cash we have"""
        self.cash.value = cash

    def getCash(self):
        """Return how much cash we have"""
        return self.cash.value
        
    def setActive(self, active):
        """Set the activity of the menu"""
        self.active = active
        if not active:
            self.selection_marker.active = False
            
    def hidePopout(self):
        """Hide the popout menu"""
        self.pop_out.visible = False
        
    def showPopout(self, theme_name, x, y):
        """Show the popout menu"""
        #
        # Put the pop out menu on the screen
        self.pop_out.visible = True
        self.pop_out.x = x
        self.pop_out.y = y-G('popout-menu-offset')
        #
        # Get turret properties
        name = G('name', theme_name)
        description = G('description', theme_name)
        rnge = G('range', theme_name)
        speed = G('speed', theme_name)
        damage = G('damage', theme_name)
        cost = G('cost', theme_name)
        #
        # Set the items
        self.pop_out.children[0].value = 'Cost: $%s' % cost
        self.pop_out.children[1].value = 'Range: %sm' % rnge
        self.pop_out.children[2].value = 'Damage: %s' % damage
        self.pop_out.children[3].value = 'Reload time: %ss' % speed
